package oemMaker;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.util.Iterator;
import java.util.Vector;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import keyboardPackage.Keyboard;

public class XMLCabType extends CabType
{
	// All the regex patterns I use.
	private static final Pattern CHECK_FOR_PARAMETER = Pattern.compile(
			"<parm.+>", Pattern.CASE_INSENSITIVE);
	private static final Pattern CHECK_FOR_SOURCE = Pattern.compile(
			"\\sname\\s*=\\s*[\"']Source[\"']", Pattern.CASE_INSENSITIVE);
	private static final Pattern CHECK_FOR_REGISTRY = Pattern.compile(
			"\\stype\\s*=\\s*[\"']Registry[\"']", Pattern.CASE_INSENSITIVE);
	private static final Pattern FIND_KEY = Pattern.compile(
			"\\sname\\s*=\\s*[\"']([^\"']+)[\"']", Pattern.CASE_INSENSITIVE);
	private static final Pattern FIND_VALUE = Pattern.compile(
			"\\svalue\\s*=\\s*[\"']([^\"']*)[\"']", Pattern.CASE_INSENSITIVE);
	private static final Pattern FIND_REGISTRY_HEADER = Pattern.compile(
			"\\stype\\s*=\\s*[\"']([^\"']*)[\"']", Pattern.CASE_INSENSITIVE);
	private static final Pattern FIND_REGISTRY_KEY = Pattern.compile(
			"\\sname\\s*=\\s*[\"']([^\"']*)[\"']", Pattern.CASE_INSENSITIVE);
	private static final Pattern FIND_REGISTRY_DATATYPE = Pattern.compile(
			"\\sdatatype\\s*=\\s*[\"']([^\"']*)[\"']", Pattern.CASE_INSENSITIVE);
	private static final Pattern FIND_REGISTRY_VALUE = Pattern.compile(
			"\\svalue\\s*=\\s*[\"']([^\"']*)[\"']", Pattern.CASE_INSENSITIVE);
	private static final Pattern CHECK_FOR_INSTALL = Pattern.compile(
			"\\stranslation\\s*=\\s*[\"']install[\"']", Pattern.CASE_INSENSITIVE);
	private static final Pattern FIND_TYPE = Pattern.compile(
			"\\stype\\s*=\\s*[\"']([^\"']+)[\"']", Pattern.CASE_INSENSITIVE);

	// Strip all the information out of the file.
	public XMLCabType(String oemName, String uuid)
	{
		super(oemName, uuid);

		try
		{
			// Try to use sed to clean the file.
			if(new File("sed.exe").exists())
			{
				// Clean the file.
				Runtime.getRuntime()
				.exec("sed -i \"s/>/>\\n/g\" " + oemName + 
						"\\orig\\_setup.xml");
				Runtime.getRuntime()
				.exec("sed -i \"/^\\$/d\" " + oemName + 
						"\\orig\\_setup.xml");
			}
			else
			{
				// Tell the user to clean the file.
				System.out.println("You do not have sed. You need to go into the program's extracted folder and then into the \"orig\" folder. Make sure _setup.xml has one tag per line and no double line returns.");
				System.out.print("When you are done checking it, press enter.");
				Keyboard.readString();
			}
		}
		// A file can't load, probably _setup.xml.
		catch (FileNotFoundException e)
		{
			System.out.println("Cannot load the setup file.");
		}
		// A file can't be written to, either the initflashfiles.txt or rgu.
		catch (IOException e)
		{
			System.out.println("Error reading the setup file.");
		}
	}

	// Process the file.
	public boolean processFile(String setupFilename)
	{
		// Load the file into the vector.
		Vector<String> xmlFile;
		
		try
		{
			BufferedReader xmlFileStream = new BufferedReader(new FileReader(setupFilename));
			xmlFile = loadFile(xmlFileStream);
			xmlFileStream.close();			
		}
		catch (FileNotFoundException e)
		{
			System.out.println("Cannot open the xml file for processing.");
			return false;
		}
		catch (IOException e)
		{
			System.out.println("Cannot read the xml file for processing.");
			return false;
		}
		
		// Run all the operations.
		getParameters(parameters, xmlFile);
		userParametersCorrection();
		getFileOperations(xmlFile);
		getRegistryEntries(xmlFile);

		// Delete the orig directory.
		deleteDir(new File(oemName + "\\orig"));
		
		// Create the option.xml.
		createOptionXML(oemName, uuid, parameters.get("AppName"));
		
		try
		{
			// Close the files.
			initflashfiles.close();
			rgu.close();			

			return true;
		}
		catch (IOException e)
		{
			System.out.println("Cannot close initflashfiles.txt or the rgu file.");
			return false;
		}
	}
	
	// Read the _setup.xml and gather all the file operations.
	private void getFileOperations(Vector<String> file)
	{
		// Loop through the file.
		Iterator<String> fileIterator = file.iterator();
		String line, directory = "";
		while (fileIterator.hasNext())
		{
			line = fileIterator.next();

			if (CHECK_FOR_INSTALL.matcher(line).find()
					&& !CHECK_FOR_PARAMETER.matcher(line).find())
			{
				// Run regex on the subject and type.
				Matcher subjectMatcher = FIND_TYPE.matcher(line);
				line = fileIterator.next();
				Matcher typeMatcher = FIND_TYPE.matcher(line);

				// Look for a subject and a type.
				if (subjectMatcher.find() && typeMatcher.find())
				{
					String type = typeMatcher.group(1);
					String subject = parameters.replaceVariables(subjectMatcher
							.group(1));

					directory = processXMLCommand(directory, type, subject,
							fileIterator);
				}
			}
		}
	}

	// Get all the parameters from the file.
	private void getParameters(Parameters parameters, Vector<String> file)
	{
		// The two matchers.
		Matcher key;
		Matcher value;

		// Loop through the file.
		Iterator<String> fileIterator = file.iterator();
		String line;
		while (fileIterator.hasNext())
		{
			// Save the line.
			line = fileIterator.next();

			// Check if this is a parameter line.
			if (CHECK_FOR_PARAMETER.matcher(line).find()
					&& !CHECK_FOR_SOURCE.matcher(line).find()
					&& !FIND_REGISTRY_DATATYPE.matcher(line).find())
			{
				// Find the key/value.
				key = FIND_KEY.matcher(line);
				key.find();
				value = FIND_VALUE.matcher(line);
				value.find();

				// Put the key and the corrected value in.
				parameters.put(key.group(1), value.group(1));
			}
		}
	}

	// Create a registry file.
	private void getRegistryEntries(Vector<String> file)
	{
		// Loop through the file until we get to the registry entries section.
		Iterator<String> fileIterator = file.iterator();
		String line;
		boolean registryFound = false;
		while (fileIterator.hasNext() && !registryFound)
		{
			line = fileIterator.next();
			registryFound = CHECK_FOR_REGISTRY.matcher(line).find();
		}

		// All the matchers.
		Matcher headerMatch;
		Matcher keyMatch;
		Matcher valueMatch;
		Matcher datatypeMatch;
		
		// Loop through the entries.
		while (fileIterator.hasNext())
		{
			// Get the next line.
			line = fileIterator.next();
			
			// Check if it's a header line.
			headerMatch = FIND_REGISTRY_HEADER.matcher(line);
			if(headerMatch.find())
			{
				System.out.println("Adding registry header: " + headerMatch.group(1));
				writeRegistryHeader(headerMatch.group(1));
			}
			// Else it should be an entry.
			else
			{
				// Check if it really is an entry.
				keyMatch = FIND_REGISTRY_KEY.matcher(line);
				valueMatch = FIND_REGISTRY_VALUE.matcher(line);
				datatypeMatch = FIND_REGISTRY_DATATYPE.matcher(line);
				
				if(keyMatch.find() &&
					valueMatch.find() &&
					datatypeMatch.find())
					
				{
					// Get the key, type, and value.
					String key = "\"" + keyMatch.group(1) + "\"";
					String type = datatypeMatch.group(1);
					String value = parameters.replaceVariables(valueMatch.group(1));
					
					System.out.println("Adding registry entry: " + key + " -> " + value);
					
					writeRegistryEntry(key, type, value);		
				}
			}
		}
	}
	
	// Load a file into a vector.
	private Vector<String> loadFile(BufferedReader file) throws IOException
	{
		Vector<String> vector = new Vector<String>();
		
		String line;
		while ((line = file.readLine()) != null)
		{
			vector.add(line);
		}
		
		return vector;
	}

	// Process an extract command.
	private void processExtractCommand(String workingDirectory, String subject,
			String target)
	{
		System.out.println("Renaming " + target + " to " + subject + ".");
		new File(oemName + "\\orig\\" + target).renameTo(new File(
				oemName + "\\" + subject));
		writeInitFlashFile(workingDirectory, "File(\"" + subject
				+ "\",\"\\Windows\\" + subject + "\")");
	}

	// Process a make directory command.
	private String processMakeDirCommand(String directoryPath)
	{
		// Seperate the directory from the directories above it.
		String topDirectory = directoryPath.substring(0, directoryPath
				.lastIndexOf("\\"));
		String directory = directoryPath.substring(directoryPath
				.lastIndexOf("\\") + 1);

		if (topDirectory.equalsIgnoreCase(""))
		{
			topDirectory = "\\";
		}

		System.out.println("Setting directory " + directory + " in directory "
				+ topDirectory + " to be created.");
		writeInitFlashFile(topDirectory, "Directory(\"" + directory + "\")");
		
		// Return a properly formatted directory string.
		if(!topDirectory.equals("\\"))
		{
			return topDirectory + "\\" + directory;			
		}
		else
		{
			return topDirectory + directory;
		}
	}

	// Process a shortcut command.
	private void processShortcutCommand(String workingDirectory, String subject,
			String target)
	{		
		// This needs to have the oemName attached, the others don't.
		createShortcut(subject, target);

		// Fix the subject if it's wrong.
		subject = subject.substring(subject.lastIndexOf("\\") + 1);
		System.out.println("Creating a shortcut named " + subject
				+ " pointing to " + target + ".");
		writeInitFlashFile(workingDirectory, "File(\"" + subject
				+ "\",\"\\Windows\\" + subject + "\")");
	}

	// Process the command given.
	private String processXMLCommand(String directory, String type,
			String subject, Iterator<String> file)
	{
		// A directory needs to be made.
		if (type.equalsIgnoreCase("MakeDir"))
		{
			directory = processMakeDirCommand(subject);
		}
		// A file needs to be renamed.
		else if (type.equalsIgnoreCase("Extract"))
		{
			Matcher valueMatcher = FIND_VALUE.matcher(file.next());
			valueMatcher.find();
			processExtractCommand(directory, subject, parameters
					.replaceVariables(valueMatcher.group(1)));
		}
		// A shortcut needs to be created.
		else if (type.equalsIgnoreCase("Shortcut"))
		{
			Matcher valueMatcher = FIND_VALUE.matcher(file.next());
			valueMatcher.find();
			processShortcutCommand(directory, oemName + "\\" + subject,
					parameters.replaceVariables(valueMatcher.group(1)));
		}
		// Something else.
		else
		{
			System.out.println("I don't know how to do \"" + type + "\" on \""
					+ subject + "\".");
		}

		return directory;
	}
	
	// Allows the user to correct the parameters.
	private void userParametersCorrection()
	{
		// Output the directions.
		System.out
				.println("I'm going to output all the parameters, and let you correct the ones you want.\n");

		// Run the loop.
		String selected;
		do
		{
			// Output the parameters.
			parameters.outputParameters();

			System.out
					.print("\nEnter the variable name you want to correct or press enter to continue: ");
			selected = Keyboard.readString();

			// Determine if they selected a valid variable.
			if (parameters.get(selected) != null)
			{
				System.out.print("Enter the new value for "
						+ parameters.get(selected) + ": ");
				parameters.set(selected, Keyboard.readString());
			}
		}
		while (!selected.equals(""));
	}
}
